﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.Core.Utils.js
//
/// <reference path=~/Scripts/jquery-1.6.2-vsdoc.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />
/// <reference path="~/Scripts/TFS/TFS.Diag.debug.js" />


TFS.module("TFS.Core.Utils", ["TFS.Diag"], function () {
    var Diag = TFS.Diag;

    function unpack(values, resultArray) {
        var i,
            j,
            prev = 0,
            value = 0,
            delta = 0,
            repeat = false,
            result = resultArray || [];

        for (i = 0; i < values.length; i++) {
            value = values[i];
            if (value === 'r') {
                if (!repeat) {
                    repeat = true;
                    continue;
                }
                else {
                    value = 2;
                    repeat = false;
                }
            }
            else if (!repeat) {
                delta = value;
                value = 1;
            }
            else {
                repeat = false;
            }

            for (j = 0; j < value; j++) {
                prev += delta;
                result[result.length] = prev;
            }
        }

        return result;
    }

    function unpackIntegerArray(array) {
        var result, i, l;

        if (array.length === 0) {
            return array;
        }

        return unpack(array);
    }

    function parseMSJSON(data, secure) {
        return Sys.Serialization.JavaScriptSerializer.deserialize(data, secure);
    }

    function stringifyMSJSON(object) {
        return Sys.Serialization.JavaScriptSerializer.serialize(object);
    }


    function findTreeNode(path, separator, comparer, textField) {
        var pos, part, children, result = null, item, i, l;

        if (path) {
            children = this.children;

            if (children && (l = children.length) > 0) {
                separator = separator || "/";

                do {
                    pos = path.indexOf(separator);

                    if (pos >= 0) {
                        part = path.substring(0, pos);
                        path = path.substring(pos + separator.length);
                    }
                    else {
                        part = path;
                        path = null;
                    }
                }
                while (path && part === separator);

                if (part) {
                    textField = textField || "text";
                    comparer = comparer || String.localeIgnoreCaseComparer;

                    for (i = 0; i < l; i++) {
                        item = children[i];
                        if (comparer(item[textField], part) === 0) {
                            result = item;
                            break;
                        }
                    }

                    if (result) {
                        if (path) {
                            if (result.children) {
                                return findTreeNode.call(result, path, separator, comparer, textField);
                            }
                            else {
                                result = null;
                            }
                        }
                    }
                }
            }
        }

        return result;
    }

    function calculateTreePath(includeRoot, separator, textField, rootField) {
        var parent = this.parent;

        rootField = rootField || "root";

        if (parent && !includeRoot && parent[rootField]) {
            parent = null;
        }

        textField = textField || "text";

        if (parent) {
            separator = separator || "/";
            return calculateTreePath.call(parent, includeRoot, separator, textField, rootField) + separator + this[textField];
        }
        else {
            return this[textField];
        }
    }

    function walkTree(f) {
        var i, l, children, item;
        if (f) {
            f.call(this, this);
            children = this.children;

            if (children && (l = children.length)) {
                for (i = 0; i < l; i++) {
                    item = children[i];
                    walkTree.call(item, f);
                }
            }
        }
    }

    function updateFilterGroups(groups, clauseNumber, insert) {
        var newGroups, gmap;
        if (groups && groups.length) {
            if (insert) {
                $.each(groups, function (i, g) {
                    if (g.end >= clauseNumber) {
                        if (g.start >= clauseNumber) {
                            g.start++;
                        }

                        g.end++;
                    }
                });
            }
            else {
                newGroups = [];
                gmap = {};

                $.each(groups, function (i, g) {
                    var gkey;
                    if (g.end >= clauseNumber) {
                        if (g.start > clauseNumber) {
                            g.start = Math.max(1, g.start - 1);
                        }

                        g.end = Math.max(1, g.end - 1);
                    }

                    if (g.start !== g.end) {
                        gkey = g.start + "_" + g.end;

                        if (!(gkey in gmap)) {
                            gmap[gkey] = true;
                            newGroups.push(g);
                        }
                    }
                });

                return newGroups;
            }
        }

        return groups;
    }
        
    function updateFilterGroupLevels(groups) {
        var maxLevel = 0;

        if (groups && groups.length) {
            $.each(groups, function (i, g) {
                g.level = 0;
            });

            function assignLevels(groups, group) {
                $.each(groups, function (i, g) {
                    if (!group) {
                        assignLevels(groups, g);
                    }
                    else if (g !== group) {
                        if (group.start <= g.start && group.end >= g.end) { //if group contains g
                            assignLevels(groups, g);
                            group.level = Math.max(group.level, g.level + 1);
                            maxLevel = Math.max(maxLevel, group.level);
                        }
                    }
                });
            }

            assignLevels(groups);
        }

        return maxLevel;
    }

    function convertValueToDisplayString(value, format) {
        /// <summary>Converts the specified value to a display string.</summary>
        /// <param name="value" type="object">The value to convert.</param>

        if (value != null) { //don't listen jslint === warning here 
        	if (typeof value === "string") {
        		return value;
        	}
			else if (value instanceof Date) {
                return value.localeFormat(format || "G");
            }
            else if (typeof value === "number") {
            	if (format) {
            		return value.localeFormat(format);
            	}
            	else {
            		return value.toDecimalLocaleString();
            	}
            }
            else if (typeof value === "boolean") {
                return value ? "True" : "False";
            }
            else {
                return value.toString();
            }
        }
        
        return "";
    }

    function isString(value) {
        /// <summary>Checks if a value is string.</summary>
        return (value != null && typeof value === "string");
    }

    function parseEmailAddressesStringToArray(emailAddressesString) {
        /// <summary>Parses a comma and/or semicolumn delimited string of email addresses into an array of the addresses.</summary>
        /// <param name="emailAddressesString" type="String">A comma and/or semicolumn delimited string of email addresses</param>
        /// <returns type="Array">The parsed array of email addresses.</returns>
        var emailAddresses = [],
            splitsOfComma, splitsOfSemicolumn;
        if (isString(emailAddressesString)) {
            splitsOfComma = emailAddressesString.split(",");
            $.each(splitsOfComma, function (i) {
                splitsOfSemicolumn = splitsOfComma[i].split(";");
                $.each(splitsOfSemicolumn, function (j) {
                    emailAddresses.push($.trim(splitsOfSemicolumn[j]));
                })
            });
        }

        return emailAddresses;
    }


    function dateUtils() {
        Diag.fail("This class is not intended to be instantiated.  All of the methods are static.");
    }

    dateUtils.extend((function () {
        var minute = 1000 * 60,
            hour = minute * 60,
            day = hour * 24,
            week = day * 7;

        return {
            MILLISECONDS_IN_MINUTE: minute,
            MILLISECONDS_IN_HOUR: hour,
            MILLISECONDS_IN_DAY: day,
            MILLISECONDS_IN_WEEK: week,

            shiftToUTC: function (date) {
                /// <summary>
                /// Shifts the date to match the UTC date.  This is done by removing the timezone offset which is applied.
                /// </summary>
                /// <param name="date" type="Date">The date to be converted.</param>
                /// <returns type="Date" />

                Diag.assertParamIsObject(date, "date");

                return new Date(date.getTime() + (date.getTimezoneOffset() * minute));
            },

            shiftToLocal: function (date) {
                /// <summary>
                /// Shifts the date to match the local date.  This is done by adding the timezone offset to the date.
                /// </summary>
                /// <param name="date" type="Date">The date to be converted.</param>
                /// <returns type="Date" />

                Diag.assertParamIsObject(date, "date");

                return new Date(date.getTime() - (date.getTimezoneOffset() * minute));
            },

            parseDateString: function (dateString, parseFormat, ignoreTimeZone) {
                /// <summary>Parses the string into a date.</summary>
                /// <param name="dateString" type="String">Date string to parse.</param>
                /// <param name="parseFormat" type="String">Optional format string to use in parsing the date. May be null or undefined</param>
                /// <param name="ignoreTimeZone" type="Boolean">
                ///     Optional value indicating to ignore the time zone set set in user preferences?
                ///     Should be set to true when a Date string should be parsed irrespective of the user's time zone (e.g. calendar control).
                /// </param>

                Diag.assertParamIsString(dateString, "dateString");
                Diag.assert(ignoreTimeZone === undefined || typeof (ignoreTimeZone) === "boolean", "ignoreTimeZone");

                var date, year;

                // Attempt to parse with the parse format.
                date = Date.parseLocale(dateString, parseFormat, ignoreTimeZone);

                // If the date could not be parsed with the parse format parse it directly.
                if (!date) {
                    date = new Date(dateString);
                }

                if (!(date instanceof Date) || isNaN(date)) {
                    // Still couldn't parse - the string doesn't represent a valid date
                    date = null;
                }
                else {
                    // Javascript parses 2-digit years as 19xx, but we want 2-digit years
                    // to map to the current century. Examine the year, and if it is in
                    // the 1900's, then correct the century unless the date was entered
                    // with the '19' prefix.
                    year = date.getFullYear();
                    if (year >= 1900 && year < 2000) {
                        if (dateString.indexOf(year) < 0) {
                            // The year is 19xx but 19xx does not appear in the date string
                            // so set the year to the current century.
                            date.setFullYear(year + Math.floor(new Date().getYear() / 100) * 100);
                        }
                    }
                }

                return date;
            },

            daysBetweenDates: function (startDate, endDate, exclusive) {
                /// <summary>Returns the number of days between the two dates. Note that any time component is ignored and the dates
                /// can be supplied in any order</summary>
                /// <param name="startDate" type="Date">The first date</param>
                /// <param name="endDate" type="Date">The second date</param>
                /// <param name="exclusive" type="Boolean" isoptional="true">If <c>true</c> then the result is exclusive of the second date (Mon->Fri==4).
                /// Otherwise the date includes the later date (Mon->Fri==5)</param>
                Diag.assertParamIsDate(startDate, "startDate");
                Diag.assertParamIsDate(endDate, "endDate");

                startDate = new Date(startDate.getTime());
                startDate.setHours(0, 0, 0, 0);

                endDate = new Date(endDate.getTime());
                endDate.setHours(0, 0, 0, 0);

                daysBetweenDates = Math.round(Math.abs(endDate.getTime() - startDate.getTime()) / day) + (exclusive === true ? 0 : 1);

                return daysBetweenDates;
            }
        };
    }()));

    return {
        parseXml: function (xml) {
            return Sys.Net.XMLDOM(xml);
        },
        unpackIntegerArray: unpackIntegerArray,
        stringifyMSJSON: stringifyMSJSON,
        parseMSJSON: parseMSJSON,
        findTreeNode: findTreeNode,
        calculateTreePath: calculateTreePath,
        walkTree: walkTree,
        updateFilterGroups: updateFilterGroups,
        updateFilterGroupLevels: updateFilterGroupLevels,
        convertValueToDisplayString: convertValueToDisplayString,
        parseEmailAddressesStringToArray: parseEmailAddressesStringToArray,
        dateUtils: dateUtils
    };
});
// SIG // Begin signature block
// SIG // MIIatwYJKoZIhvcNAQcCoIIaqDCCGqQCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFOdBIM0FWcT1
// SIG // HuaDkmG5qzS8pULXoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggShMIIEnQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIG6MBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBQKuxhCsKvc
// SIG // DnDOiJSY3+TB5FC8dTBaBgorBgEEAYI3AgEMMUwwSqAw
// SIG // gC4AVABGAFMALgBDAG8AcgBlAC4AVQB0AGkAbABzAC4A
// SIG // ZABlAGIAdQBnAC4AagBzoRaAFGh0dHA6Ly9taWNyb3Nv
// SIG // ZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBABokd4BqdcTK
// SIG // xwBbd0CrLC24bOl48ExFrVklJukGuGJ9Mqr5gRq90ijn
// SIG // OyQ1++uwcQyZj5Cq5vI2X1SYJ9Px9bGliwEt/PSh5RE9
// SIG // G7uu0eyBsTGpegcorm1ThIEjjWMYebWPpnMWmvwOyGqx
// SIG // T65d2f5rHRkNsY4GJAEC4Cbr3orCaEKexPUUGzfTX6rm
// SIG // 4XQEBibUBK8zGeay1c948riMaPZvjAlXS9lSg7Vq15Wr
// SIG // m3kbSgTIG+t0Yyd95OVrZWXkJDqcMXauII8yW+0TYhLD
// SIG // NiRoHriUXKvcm8AUly/DCw81ovlkLUPEWC1ikWP6cEB3
// SIG // QQB0w4xnItja5UE3MLDZxImhggIoMIICJAYJKoZIhvcN
// SIG // AQkGMYICFTCCAhECAQEwgY4wdzELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRpbWUtU3Rh
// SIG // bXAgUENBAhMzAAAAKzkySMGyyUjzAAAAAAArMAkGBSsO
// SIG // AwIaBQCgXTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcB
// SIG // MBwGCSqGSIb3DQEJBTEPFw0xMzAzMTUwNjMzNTVaMCMG
// SIG // CSqGSIb3DQEJBDEWBBTWSkDXN5AeLJzphCqbh9JI5Hhz
// SIG // bjANBgkqhkiG9w0BAQUFAASCAQA1Rk0WEHFW2zifkrAu
// SIG // S33UI/g4eHDoi6q6wlPPwEIst3TnhNHctxbNB06WKtVV
// SIG // W8wS+MDs7hmmdl/P1DxAWTEqPPpWwnkQDx+gZS1Kmgcz
// SIG // 88C1kxHqhQRelUl85K95tNdGrIKe5sSyTQkiP8f6Hq7s
// SIG // IFDloilN1SLcmGyc9hiG1w/CsUrm4mJVzrIMUsSkGIoz
// SIG // FWuOurIQE/HPIiSCeu3+k7K1Pzpg/JKGuHR0CgELdGas
// SIG // rnoJkAmNNNBb/h/G7lSvk2xF+Wpppk4w8EJn6MjpEr8k
// SIG // VVdrIu887Mizp0ioOgr8u+xbnMmB3Z2zOKziJOT6J1Ye
// SIG // wKiaPCD3z0lJ9AZd
// SIG // End signature block
